<?php
/*
 * jQuery File Upload Plugin PHP Class 6.1.2
 * https://github.com/blueimp/jQuery-File-Upload
 *
 * Copyright 2010, Sebastian Tschan
 * https://blueimp.net
 *
 * Licensed under the MIT license:
 * http://www.opensource.org/licenses/MIT
 */

class UploadHandler
{
	protected $options;

	function __construct($options = null, $initialize = true)
	{
		$this->options = array(
			'script_url' => '',
			'upload_dir' => '',
			'user_dirs' => false,
			'file_id' => '',
			'mkdir_mode' => 0755,
			'delete_type' => 'POST',
			
			// Defines which files (based on their names) are accepted for upload:
			'accept_file_types' => '/.+$/i',
			
			'max_number_of_files' => null,
		);
		if ($options)
		{
			$this->options = array_merge($this->options, $options);
		}
		if ($initialize)
		{
			$this->initialize();
		}
	}

	protected function initialize()
	{
		switch ($_SERVER['REQUEST_METHOD'])
		{
			case 'OPTIONS':
			case 'HEAD':
				$this->head();
				break;
			case 'GET':
				$this->get();
				break;
			case 'PATCH':
			case 'PUT':
			case 'POST':
				$this->post();
				break;
			case 'DELETE':
				$this->delete();
				break;
			default:
				header('HTTP/1.1 405 Method Not Allowed');
		}
	}

	protected function get_user_path()
	{
		if ($this->options['user_dirs'] AND $this->options['file_id'])
		{
			return $this->options['file_id'] . '-';
		}
		return '';
	}

	protected function get_upload_path($file_name = null, $version = null, $altUploadDir = null)
	{
		return (empty($altUploadDir) ? $this->options['upload_dir'] : $altUploadDir . '/') . $this->get_user_path() . (empty($version) ? '' : $version) . ($file_name ? $file_name : '');
	}

	protected function get_query_separator($url)
	{
		return strpos($url, '?') === false ? '?' : '&';
	}

	protected function set_file_delete_properties($file)
	{
		$file->delete_type = $this->options['delete_type'];		
		$file->delete_url = $this->options['script_url'] . $this->get_query_separator($this->options['script_url']) . 'file=' . rawurlencode($file->name) . '&_method=DELETE&fileid=' . $file->fileid;
	}

	// Fix for overflowing signed 32 bit integers,
	// works for sizes up to 2^32-1 bytes (4 GiB - 1):
	protected function fix_integer_overflow($size)
	{
		if ($size < 0)
		{
			$size += 2.0 * (PHP_INT_MAX + 1);
		}
		return $size;
	}

	protected function get_file_size($file_path, $clear_stat_cache = false)
	{
		if ($clear_stat_cache)
		{
			clearstatcache(true, $file_path);
		}
		return $this->fix_integer_overflow(filesize($file_path));
	}

	protected function is_valid_file_object($file_name, $fileInfo)
	{
		$file_path = $this->get_upload_path($file_name, null, $fileInfo['filepath']);
		if (is_file($file_path) AND $file_name[0] !== '.')
		{
			return true;
		}
		return false;
	}

	protected function get_file_object($file_name, $fileInfo)
	{
		if ($this->is_valid_file_object($file_name, $fileInfo))
		{
			$file = new stdClass();
			$file->name = preg_replace_callback("/(&#[0-9]+;)/", 'reverseEntities', $fileInfo['filename']);
			$file->fileid = $fileInfo['fileid'];
			$file->screenshot_checked = $fileInfo['screenshot'] ? ' checked="checked"' : '';
			$file->thumbnail_checked = $fileInfo['thumbnail'] ? ' checked="checked"' : '';
			$file->size = $this->get_file_size(
				$this->get_upload_path($file_name, null, $fileInfo['filepath'])
			);
			$this->set_file_delete_properties($file);

			return $file;
		}
		return null;
	}

	protected function get_file_objects($iteration_method = 'get_file_object')
	{
		$upload_dir = $this->get_upload_path();
		if (!is_dir($upload_dir))
		{
			return array();
		}

		if ($_REQUEST['downloadid'])
		{
			// We have a download ID, grab files in download
			$fileList = VBDOWNLOADS::$db->fetchAll('
				SELECT *
				FROM $dbtech_downloads_file
				WHERE downloadid = ?
			', array(
				$_REQUEST['downloadid']
			));
		}
		else if ($_REQUEST['fileid'])
		{
			// We have a file ID, grab other files in its download
			$fileList = VBDOWNLOADS::$db->fetchAll('
				SELECT *
				FROM $dbtech_downloads_file
				WHERE downloadid = (
					SELECT downloadid
					FROM $dbtech_downloads_file
					WHERE fileid = ?
				)
			', array(
				$_REQUEST['fileid']
			));
		}
		else
		{
			return array();
		}

		$files = array();
		foreach ($fileList as &$file)
		{
			if ($file['filepath'])
			{
				if (substr($file['filepath'], 0, 2) == './')
				{
					// Prepare this
					$file['filepath'] = str_replace('./', DIR . '/', $file['filepath']);
				}

				// Imported file
				$files[] = $file['filename'];
			}
			else
			{
				// Normal file
				$files[] = $file['fileid'] . '-' . $file['filename'];
			}
		}

		return array_values(array_filter(array_map(
			array($this, $iteration_method),
			$files,
			$fileList
		)));
	}

	protected function get_error_message($error)
	{
		global $vbphrase;

		return array_key_exists('dbtech_downloads_uploaderror_' . $error, $vbphrase) ?
			$vbphrase['dbtech_downloads_uploaderror_' . $error] : $error;
	}

	function get_config_bytes($val)
	{
		$val = trim($val);
		$last = strtolower($val[strlen($val)-1]);
		switch($last)
		{
			case 'g':
				$val *= 1024;
			case 'm':
				$val *= 1024;
			case 'k':
				$val *= 1024;
		}
		return $this->fix_integer_overflow($val);
	}

	protected function validate($uploaded_file, $file, $error, $index)
	{
		if ($error)
		{
			$file->error = $this->get_error_message($error);
			return false;
		}
		$content_length = $this->fix_integer_overflow(intval($_SERVER['CONTENT_LENGTH']));
		$post_max_size = $this->get_config_bytes(ini_get('post_max_size'));
		if ($post_max_size AND ($content_length > $post_max_size))
		{
			$file->error = $this->get_error_message('post_max_size');
			return false;
		}
		if (!preg_match($this->options['accept_file_types'], $file->name))
		{
			$file->error = $this->get_error_message('accept_file_types');
			return false;
		}
		if ($uploaded_file AND is_uploaded_file($uploaded_file))
		{
			$file_size = $this->get_file_size($uploaded_file);
		}
		else
		{
			$file_size = $content_length;
		}

		if (is_int($this->options['max_number_of_files']) AND (count($this->get_file_objects('is_valid_file_object')) >= $this->options['max_number_of_files']))
		{
			$file->error = $this->get_error_message('max_number_of_files');
			return false;
		}

		$ext = strtolower(pathinfo($file->name, PATHINFO_EXTENSION));
		list($img_width, $img_height) = @getimagesize($uploaded_file);

		foreach (VBDOWNLOADS::$cache['extension'] as $extension)
		{
			if ($extension['extension'] != $ext)
			{
				// Not the ext we're looking for
				continue;
			}

			if ($extension['maxsize'] AND (
				$file_size > $extension['maxsize'] OR
				$file->size > $extension['maxsize']
			))
			{
				$file->error = $this->get_error_message('max_file_size');
				return false;
			}

			if (is_int($img_width))
			{
				// Check width
				if ($extension['maxwidth'] AND $img_width > $extension['maxwidth'])
				{
					$file->error = $this->get_error_message('max_width');
					return false;
				}

				if ($extension['maxheight'] AND $img_height > $extension['maxheight'])
				{
					$file->error = $this->get_error_message('max_height');
					return false;
				}
			}
		}
		return true;
	}

	protected function upcount_name_callback($matches)
	{
		$index = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
		$ext = isset($matches[2]) ? $matches[2] : '';
		return ' (' . $index . ')'.$ext;
	}

	protected function upcount_name($name)
	{
		return preg_replace_callback(
			'/(?:(?: \(([\d]+)\))?(\.[^.]+))?$/',
			array($this, 'upcount_name_callback'),
			$name,
			1
		);
	}

	protected function get_unique_filename($name, $type, $index, $content_range)
	{
		while(is_dir($this->get_upload_path($name)))
		{
			$name = $this->upcount_name($name);
		}
		
		// Keep an existing filename if this is part of a chunked upload:
		$uploaded_bytes = $this->fix_integer_overflow(intval($content_range[1]));
		while (is_file($this->get_upload_path($name)))
		{
			if ($uploaded_bytes === $this->get_file_size($this->get_upload_path($name)))
			{
				break;
			}
			$name = $this->upcount_name($name);
		}
		return $name;
	}

	protected function trim_file_name($name, $type, $index, $content_range)
	{
		// Remove path information and dots around the filename, to prevent uploading
		// into different directories or replacing hidden system files.
		// Also remove control characters and spaces (\x00..\x20) around the filename:
		$name = trim(basename(stripslashes($name)), ".\x00..\x20");
		
		// Use a timestamp for empty filenames:
		if (!$name)
		{
			$name = str_replace('.', '-', microtime(true));
		}
		
		// Add missing file extension for known image types:
		if (strpos($name, '.') === false AND preg_match('/^image\/(gif|jpe?g|png)/', $type, $matches))
		{
			$name .= '.' . $matches[1];
		}

		$info = pathinfo($name);
		return $info['filename'] . (isset($info['extension']) ? '.' . strtolower($info['extension']) : '');
	}

	protected function get_file_name($name, $type, $index, $content_range)
	{
		return $this->get_unique_filename(
			$this->trim_file_name($name, $type, $index, $content_range),
			$type,
			$index,
			$content_range
		);
	}

	protected function handle_form_data(&$file, $index, $uploaded_file)
	{
		global $vbulletin;

		if ($file->size != $this->get_file_size($uploaded_file))
		{
			// Not ready yet
			return;
		}

		if ($_REQUEST['downloadid'])
		{
			if (!$download = VBDOWNLOADS::$db->fetchRow('
				SELECT *
				FROM $dbtech_downloads_download
				WHERE downloadid = ?
			', array(
				$_REQUEST['downloadid']
			)))
			{
				// We don't even care.
				return;
			}

			// Update the download
			$dm =& VBDOWNLOADS::initDataManager('Download', $vbulletin, ERRTYPE_SILENT);
				$dm->set_existing($download);
				$dm->set('numfiles', 'numfiles + 1', false);
			$dm->save();
		}
		else
		{
			$download = array(
				'downloadid' => 0,
				'userid' => $vbulletin->userinfo['userid']
			);
		}

		$ext = strtolower(pathinfo($file->name, PATHINFO_EXTENSION));
		$file->image = 0;
		foreach (VBDOWNLOADS::$cache['extension'] as $extension)
		{
			if ($extension['extension'] == $ext)
			{
				// Check if extension is image
				$file->image = (int)$extension['isimage'];
				break;
			}
		}

		// Set file hash
		$file->hash = md5($uploaded_file);

		// Create the file in the DM
		$dm =& VBDOWNLOADS::initDataManager('File', $vbulletin, ERRTYPE_SILENT);
			$dm->set('filename', $file->name);
			$dm->set('hashkey', $file->hash);
			$dm->set('filesize', $file->size);
			$dm->set('image', $file->image);
			$dm->set('userid', $download['userid']);
			$dm->set('downloadid', $download['downloadid']);
		$file->fileid = $dm->save();

		// Prepend file ID to file name
		$file->name = ($file->fileid ? $file->fileid . '-' : '') . $file->name;
	}

	protected function charset_decode_utf_8($string)
	{ 
		/* Only do the slow convert if there are 8-bit characters */ 
		/* avoid using 0xA0 (\240) in preg_match ranges. RH73 does not like that */ 
		if (!preg_match("/[\200-\237]/", $string) and !preg_match("/[\241-\377]/", $string))
		{
			return $string; 
		}

		// decode three byte unicode characters 
		$string = preg_replace_callback(
			"/([\340-\357])([\200-\277])([\200-\277])/", 
			array($this, 'decodeThreeByte'), 
			$string
		);
		
		// decode two byte unicode characters 
		$string = preg_replace_callback(
			"/([\300-\337])([\200-\277])/", 
			array($this, 'decodeTwoByte'), 
			$string
		);

		return $string; 
	}

	protected function decodeThreeByte($matches)
	{
		return '&#' . ((ord($matches[1]) - 224) * 4096 + (ord($matches[2]) - 128) * 64 + (ord($matches[3]) - 128)) . ';';
	}

	protected function decodeTwoByte($matches)
	{
		return '&#' . ((ord($matches[1]) - 192) * 64 + (ord($matches[2]) - 128)) . ';';
	}

	protected function handle_file_upload($uploaded_file, $name, $size, $type, $error, $index = null, $content_range = null)
	{
		$origName = $name;
		//$name = to_charset($name, 'utf-8');
		$name = $this->charset_decode_utf_8($name);

		$file = new stdClass();
		$file->name = $this->get_file_name($name, $type, $index, $content_range);
		$file->size = $this->fix_integer_overflow(intval($size));
		$file->type = $type;
		$file->screenshot_checked = '';
		$file->thumbnail_checked = '';
		if ($this->validate($uploaded_file, $file, $error, $index))
		{
			$this->handle_form_data($file, $index, $uploaded_file);
			$upload_dir = $this->get_upload_path();
			if (!is_dir($upload_dir))
			{
				mkdir($upload_dir, $this->options['mkdir_mode'], true);
			}			
			$file_path = $this->get_upload_path($file->name);
			$append_file = $content_range AND is_file($file_path) AND $file->size > $this->get_file_size($file_path);
			if ($uploaded_file AND is_uploaded_file($uploaded_file))
			{
				// multipart/formdata uploads (POST method uploads)
				if ($append_file)
				{
					file_put_contents(
						$file_path,
						fopen($uploaded_file, 'r'),
						FILE_APPEND
					);
				}
				else
				{
					move_uploaded_file($uploaded_file, $file_path);
				}
			}
			else
			{
				// Non-multipart uploads (PUT method support)
				file_put_contents(
					$file_path,
					fopen('php://input', 'r'),
					$append_file ? FILE_APPEND : 0
				);
			}
			$file_size = $this->get_file_size($file_path, $append_file);
			if ($file_size === $file->size)
			{
				// Nothin
			}
			else if (!$content_range)
			{
				@unlink($file_path);
				$file->error = 'abort';
			}
			$file->size = $file_size;
			$file->name = $origName;
			$this->set_file_delete_properties($file);
		}
		return $file;
	}
	
	protected function generate_response($content, $print_response = true)
	{
		if ($print_response)
		{
			$json = json_encode($content);
			$redirect = isset($_REQUEST['redirect']) ?
				stripslashes($_REQUEST['redirect']) : null;
			if ($redirect)
			{
				header('Location: '.sprintf($redirect, rawurlencode($json)));
				return;
			}
			$this->head();
			if (isset($_SERVER['HTTP_CONTENT_RANGE']))
			{
				$files = isset($content['files']) ?
					$content['files'] : null;
				if ($files AND is_array($files) AND is_object($files[0]) AND $files[0]->size)
				{
					header('Range: 0-' . ($this->fix_integer_overflow(intval($files[0]->size)) - 1));
				}
			}

			echo($json);
		}
		return $content;
	}

	protected function get_file_name_param()
	{
		return isset($_GET['file']) ? basename(stripslashes($_GET['file'])) : null;
	}

	public function head()
	{
		header('Pragma: no-cache');
		header('Cache-Control: no-store, no-cache, must-revalidate');
		header('Content-Disposition: inline; filename="files.json"');
		// Prevent Internet Explorer from MIME-sniffing the content-type:
		header('X-Content-Type-Options: nosniff');
		header('Vary: Accept');
		if (isset($_SERVER['HTTP_ACCEPT']) AND (strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false))
		{
			header('Content-type: application/json');
		}
		else
		{
			header('Content-type: text/plain');
		}
	}

	public function get($print_response = true)
	{
		$file_name = $this->get_file_name_param();
		if ($file_name)
		{
			$response = array(
				'file' => $this->get_file_object($file_name)
			);
		}
		else
		{
			$response = array(
				'files' => $this->get_file_objects()
			);
		}
		return $this->generate_response($response, $print_response);
	}

	public function post($print_response = true)
	{
		if (isset($_REQUEST['_method']) AND $_REQUEST['_method'] === 'DELETE')
		{
			return $this->delete($print_response);
		}

		$upload = isset($_FILES['files']) ? $_FILES['files'] : null;
		
		// Parse the Content-Disposition header, if available:
		$file_name = isset($_SERVER['HTTP_CONTENT_DISPOSITION']) ?
			rawurldecode(preg_replace(
				'/(^[^"]+")|("$)/',
				'',
				$_SERVER['HTTP_CONTENT_DISPOSITION']
			)) : null;
		
		// Parse the Content-Range header, which has the following form:
		// Content-Range: bytes 0-524287/2000000
		$content_range = isset($_SERVER['HTTP_CONTENT_RANGE']) ? preg_split('/[^0-9]+/', $_SERVER['HTTP_CONTENT_RANGE']) : null;
		
		$size =  $content_range ? $content_range[3] : null;
		$files = array();
		
		if ($upload AND is_array($upload['tmp_name']))
		{
			// param_name is an array identifier like "files[]",
			// $_FILES is a multi-dimensional array:
			foreach ($upload['tmp_name'] as $index => $value)
			{
				$files[] = $this->handle_file_upload(
					$upload['tmp_name'][$index],
					$file_name ? $file_name : $upload['name'][$index],
					$size ? $size : $upload['size'][$index],
					$upload['type'][$index],
					$upload['error'][$index],
					$index,
					$content_range
				);
			}
		}
		else
		{
			// param_name is a single object identifier like "file",
			// $_FILES is a one-dimensional array:
			$files[] = $this->handle_file_upload(
				isset($upload['tmp_name']) ? $upload['tmp_name'] : null, 
				$file_name ? $file_name : (isset($upload['name']) ? $upload['name'] : null),
				$size ? $size : (isset($upload['size']) ? $upload['size'] : $_SERVER['CONTENT_LENGTH']),
				isset($upload['type']) ? $upload['type'] : $_SERVER['CONTENT_TYPE'],
				isset($upload['error']) ? $upload['error'] : null,
				null,
				$content_range
			);
		}

		return $this->generate_response(
			array('files' => $files),
			$print_response
		);
	}

	public function delete($print_response = true)
	{
		global $vbulletin;

		if (!$file = VBDOWNLOADS::$db->fetchRow('
			SELECT *
			FROM $dbtech_downloads_file
			WHERE fileid = ?
		', array(
			$_REQUEST['fileid']
		)))
		{
			return $this->generate_response(array('success' => false), $print_response);
		}

		if (!$download = VBDOWNLOADS::$db->fetchRow('
			SELECT *
			FROM $dbtech_downloads_download
			WHERE downloadid = ?
		', array(
			$file['downloadid']
		)))
		{
			return $this->generate_response(array('success' => false), $print_response);
		}

		$file_name = $file['fileid'] . '-' . $file['filename'];
		$file_path = $this->get_upload_path($file_name, null, $file['filepath']);
		$success = is_file($file_path) AND $file_name[0] !== '.';
		if (!$file['filepath'])
		{
			// Only unlink files without file path
			$success = $success AND @unlink($file_path);
		}

		if ($success)
		{
			// Update the download
			$dm =& VBDOWNLOADS::initDataManager('Download', $vbulletin, ERRTYPE_SILENT);
				$dm->set_existing($download);
				$dm->set('numfiles', 'numfiles - 1', false);
			$dm->save();

			// Update the file
			$dm =& VBDOWNLOADS::initDataManager('File', $vbulletin, ERRTYPE_SILENT);
				$dm->set_existing($file);
			$dm->delete();
		}
		return $this->generate_response(array('success' => $success), $print_response);
	}

}